/* Operators.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.engine.apengine;

import org.grinvin.expr.Compound;
import org.grinvin.expr.Expression;
import org.grinvin.expr.IntegerLiteral;
import org.grinvin.expr.StandardOperator;


public class Operators {
    
    //
    public interface Operator {
        public String renderString(String[] args);
        
        public Expression toExpression (Expression[] args);
    }
    
    //
    public enum UnaryOperator implements Operator {
        MINUSONE("-1") {
            public double eval(double x) { return x - 1; }
            public String renderString(String[] args) { return args[0] + " - 1"; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.MINUS, args[0], IntegerLiteral.ONE);
            }
        },
        PLUSONE("+1") {
            public double eval(double x) { return x + 1; }
            public String renderString(String[] args) { return args[0] + " + 1"; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.PLUS, args[0], IntegerLiteral.ONE);
            }
        },
        TIMESTWO("*2") {
            public double eval(double x) { return x * 2; }
            public String renderString(String[] args) { return args[0] + " * 2"; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.TIMES, IntegerLiteral.TWO, args[0]);
            }
        },
        DIVTWO("/2") {
            public double eval(double x) { return x / 2; }
            public String renderString(String[] args) { return args[0] + " / 2"; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.DIV, args[0], IntegerLiteral.TWO);
            }
        },
        SQUARED("^2") {
            public double eval(double x) { return x * x; }
            public String renderString(String[] args) { return args[0] + " ^ 2"; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.POW, args[0], IntegerLiteral.TWO);
            }
        },
        NEGATE("-") {
            public double eval(double x) { return -x; }
            public String renderString(String[] args) { return "-" + args[0]; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.NEG, args[0]);
            }
        },
        ONEOVER("1/") {
            public double eval(double x) {
                if(almostZero(x))
                    return Double.NaN;
                else
                    return 1 / x;
            }
            public String renderString(String[] args) { return "1 / " +  args[0]; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.DIV, IntegerLiteral.ONE, args[0]);
            }
        };
        
        private String name;
        
        UnaryOperator(String name) {
            this.name = name;
        }
        
        public String toString() {
            return name;
        }
        
        public abstract double eval(double x);
    }
    
    //
    public enum BinaryOperator implements Operator {
        PLUS("+", true) { 
            public double eval(double x, double y) { return x + y; }
            public String renderString(String[] args) { return args[0] + " + " + args[1]; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.PLUS, args);
            }
        },
        MINUS("-", false) { 
            public double eval(double x, double y) { return x - y; }
            public String renderString(String[] args) { return args[0] + " - " + args[1]; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.MINUS, args);
            }
        },
        TIMES("*", true) { 
            public double eval(double x, double y) { return x * y; }
            public String renderString(String[] args) { return args[0] + " * " + args[1]; }
            public Expression toExpression(Expression[] args) {
               return new Compound (StandardOperator.TIMES, args);
            }
        },
        DIVIDE("/", false) {
            public double eval(double x, double y) {
                if(almostZero(y))
                    return Double.NaN;
                else
                    return x / y;
            }
            public String renderString(String[] args) { return args[0] + " / " + args[1]; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.DIV, args);
            }
        },
        POWER("^", false) { 
            public double eval(double x, double y) { return Math.pow(x, y); }
            public String renderString(String[] args) { return args[0] + " ^ " + args[1]; }
            public Expression toExpression(Expression[] args) {
                return new Compound (StandardOperator.POW, args);
            }
        };
        
        private String name;
        
        private boolean commutative;
        
        BinaryOperator(String name, boolean commutative) {
            this.name = name;
            this.commutative = commutative;
        }
        
        public String toString() {
            return name;
        }
        
        public abstract double eval(double x, double y);
        
        public boolean isCommutative() {
            return commutative;
        }
    }
    
    //
    public static class Invariant implements Operator {
        
        public int number;
        
        public Invariant(int number) {
            this.number = number;
        }
        
        public String toString() {
            return "I" + number;
        }
        
        public String renderString(String[] args) {
            return toString();
        }

        public Expression toExpression(Expression[] args) {
            return null; // NOT USED
        }
    }
    
    //
    private static boolean almostZero(double value) {
        return Math.abs(value) < (1e-6);
    }
    
}
